//
//  FJDItemsInFolderWindowController_DragAndDrop.m
//  FJDItemsInFolderWindow
//
//  Created by FUJIDANA on 06/08/18.
//  Copyright 2006 FUJIDDANA. All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions
// are met:
// 1. Redistributions of source code must retain the above copyright
//    notice, this list of conditions and the following disclaimer.
// 2. Redistributions in binary form must reproduce the above copyright
//    notice, this list of conditions and the following disclaimer in the
//    documentation and/or other materials provided with the distribution.
//
// THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
// IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
// OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
// IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
// INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
// NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
// DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
// THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
// THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.


#import "FJDItemsInFolderWindowController.h"
#import "FJDSortedArrayController.h"

NSString *FJDObjectIDsPboardType = @"FJDObjectIDsPboardType";


@interface FJDItemsInFolderWindowController (DragAndDrop_Private)

- (NSArray *)URIRepresentationsOfManagedObjects:(NSArray *)objects;
- (NSArray *)managedObjectsOfURIRepresentations:(NSArray *)URIReps;
- (void)rearrangeItemsWithUndoRegistration;

@end


#pragma mark -


@implementation FJDItemsInFolderWindowController (DnD)


#pragma mark Methods to make drag and drop operation-ready

- (void)setupForDragAndDropOperation
{
	[itemsTableView registerForDraggedTypes:[self readablePasteboardTypes]];
	[itemsTableView setDataSource:self];
	
	[foldersTableView registerForDraggedTypes:[self readablePasteboardTypes]];
	[foldersTableView setDataSource:self];
}

#pragma mark Acceptable pasteboard type

// --- Subclass can override this method.

- (NSArray *)readablePasteboardTypes
{
	return [NSArray arrayWithObject:FJDObjectIDsPboardType];
}

#pragma mark Methods implementing NSTableDataSource informal protocol

// --- Do nothing since key-value binding shows data in table view, instead of this method.

- (int)numberOfRowsInTableView:(NSTableView *)tableView
{
	return 0;
}

// --- Do nothing since key-value binding shows data in table view, instead of this method.

- (id)tableView:(NSTableView *)tableView objectValueForTableColumn:(NSTableColumn *)tableColumn row:(int)rowIndex
{
	return nil;
}


- (BOOL)tableView:(NSTableView *)tableView writeRowsWithIndexes:(NSIndexSet *)rowIndexes toPasteboard:(NSPasteboard *)pboard
{
	if (tableView == itemsTableView) {
	
		// Write URL representations of selected object to pasteboard.
		
		// Declare types
		[pboard declareTypes:[NSArray arrayWithObject:FJDObjectIDsPboardType] owner:nil];
		
		// Get dragged objects and convert each object to URIRepresentation (NSString)
		NSArray *objects = [[itemsController arrangedObjects] objectsAtIndexes:rowIndexes];
		NSArray *URIReps = [self URIRepresentationsOfManagedObjects:objects];
		if (URIReps == nil || [URIReps count] == 0) {
			return NO;
		}
		
		// Set URI representations
		[pboard setPropertyList:URIReps forType:FJDObjectIDsPboardType];
		
		return YES;
	}
	return NO;
}

- (NSDragOperation)tableView:(NSTableView *)tableView validateDrop:(id <NSDraggingInfo>)info proposedRow:(int)row proposedDropOperation:(NSTableViewDropOperation)operation
{
	if (operation == NSTableViewDropOn 
		&& tableView == foldersTableView
		&& [info draggingSource] == itemsTableView
		&& [[[info draggingPasteboard] types] containsObject:FJDObjectIDsPboardType]) {
		
		// Dragged from itemsTableView to folderTableView... (This means folder properties of dragged items is being changed.)
		return NSDragOperationMove;
		
	} else if (operation == NSTableViewDropAbove
			   && tableView == itemsTableView
			   && [info draggingSource] == itemsTableView
			   && [itemsController filterPredicate] == nil
			   && [[[info draggingPasteboard] types] containsObject:FJDObjectIDsPboardType] 
			   && [itemsController isSortedUsingDefaultDescriptors]) {
			   
		// Dragged between itemsTableView... (This means dragged items is being reordered.)
		return NSDragOperationMove;
		
	}
	return NSDragOperationNone;
}


- (BOOL)tableView:(NSTableView *)tableView acceptDrop:(id <NSDraggingInfo>)info row:(int)row dropOperation:(NSTableViewDropOperation)operation
{
	if (operation == NSTableViewDropOn 
		&& tableView == foldersTableView
		&& [info draggingSource] == itemsTableView
		&& [[[info draggingPasteboard] types] containsObject:FJDObjectIDsPboardType]) {
	
		// Dragged from itemsTableView to folderTableView... (This means folder properties of dragged items is being changed.)
		
		if ([[foldersController arrangedObjects] count] <= row) return NO;
		NSManagedObject *folder = [[foldersController arrangedObjects] objectAtIndex:row];
		
		NSArray *URIReps = [[info draggingPasteboard] propertyListForType:FJDObjectIDsPboardType];
		if (URIReps == nil || [URIReps isKindOfClass:[NSArray class]] == NO || [URIReps count] == 0) return NO;
		
		NSEnumerator *enumerator = [[self managedObjectsOfURIRepresentations:URIReps] objectEnumerator];
		NSManagedObject *object;
		while (object = [enumerator nextObject]) {
			[object setValue:folder forKey:@"folder"];
		}
		
		[self rearrangeItemsWithUndoRegistration];
		
		return YES;
		
	} else if (operation == NSTableViewDropAbove
			   && tableView == itemsTableView
			   && [info draggingSource] == itemsTableView
			   && [itemsController filterPredicate] == nil
			   && [[[info draggingPasteboard] types] containsObject:FJDObjectIDsPboardType] 
			   && [itemsController isSortedUsingDefaultDescriptors]) {
			   
		// Dragged between itemsTableView... (This means dragged items is being reordered.)
		
		NSArray *URIReps = [[info draggingPasteboard] propertyListForType:FJDObjectIDsPboardType];
		if (URIReps == nil || [URIReps isKindOfClass:[NSArray class]] == NO || [URIReps count] == 0) return NO;
		NSArray *objects = [self managedObjectsOfURIRepresentations:URIReps];
		
		[itemsController reorderByMovingObjects:objects toArrangedObjectIndex:row];
		
		[self rearrangeItemsWithUndoRegistration];
		
		return YES;
	}
	return NO;
}

@end


#pragma mark -


@implementation FJDItemsInFolderWindowController (DragAndDrop_Private)

// --- get NSArray of URI Representations from NSArray of managed objects. ---

- (NSArray *)URIRepresentationsOfManagedObjects:(NSArray *)objects
{
	NSMutableArray	*URIReps = [NSMutableArray arrayWithCapacity:[objects count]];
	NSEnumerator	*enumerator = [objects objectEnumerator];
	NSManagedObject	*object;
	
	while (object = [enumerator nextObject]) {
		
		// Get URI representations coresponding to current object (NSManagedObject).
		NSString* URIRep = [[[object objectID] URIRepresentation] absoluteString];
		
		if (URIRep == nil) continue;
		
		// Add URI representation
		[URIReps addObject:URIRep];
	}
	return URIReps;
}

// --- get NSArray of managed objects from NSArray of URI Representations. ---

- (NSArray *)managedObjectsOfURIRepresentations:(NSArray *)URIReps
{
	NSManagedObjectContext			*context		= [self managedObjectContext];
	NSPersistentStoreCoordinator	*coordinator	= [self persistentStoreCoordinator];
	
	NSMutableArray		*objects	= [NSMutableArray arrayWithCapacity:[URIReps count]];
	NSEnumerator		*enumerator	= [URIReps objectEnumerator];
	NSString			*URIRep;
	NSManagedObjectID	*objectID;
	NSManagedObject		*object;
	
	while (URIRep = [enumerator nextObject]) {
		// get an object ID coresponding to current object (NSString representing URI Representaion).
		objectID = [coordinator managedObjectIDForURIRepresentation:[NSURL URLWithString:URIRep]];
		if (objectID == nil) continue;
		
		// get an object corespponding to the object ID obtained above.
		object = [context objectWithID:objectID];
		if (object == nil) continue;
		
		// add object into array.
		[objects addObject:object]; 
	}
	return objects;
}

// --- rearrange itemsController. invoked when user changed ``order'' property of items by drag and drop. ---

- (void)rearrangeItemsWithUndoRegistration
{
	NSUndoManager *undoManager = [[self managedObjectContext] undoManager];
	[[undoManager prepareWithInvocationTarget:self] rearrangeItemsWithUndoRegistration];
	[itemsController rearrangeObjects];
}

@end
